// LPS_Demo.cpp : This file contains the 'main' function. Program execution begins and ends there.
// RD 1/15/2025 derived from LDA64Test.cpp
// NB 6/11/2025 Adapted for LPS devices
//
//	This program is a demonstration of how to use the multiple channel SetPhaseAngle function
//

#include "stdafx.h"
#include "vnx_LPS_api.h"

// ------------------------------ - Equates-----------------------------------------------
#define CL_SWP_DIRECTION		0x04	// MASK: bit = 0 for ramp up, 1 for ramp down 
#define CL_SWP_CONTINUOUS		0x02	// MASK: bit = 1 for continuous ramping
#define CL_SWP_ONCE				0x01	// MASK: bit = 1 for single ramp
#define CL_SWP_BIDIRECTIONALLY	0x10	// MASK: bit = 1 for bidirectional ramps (V2 LPS's only)


// ------------------------------- Allocations -------------------------------------------

static DEVID MyDevices[MAXDEVICES];				// I have statically allocated this array for convenience
												// It holds a list of device IDs for the connected devices
												// They are stored starting at MyDevices[0]

static char MyDeviceNameA[MAX_MODELNAME];		// NB -- this is a single byte char array for testing the ASCII name function
static wchar_t MyDeviceNameW[MAX_MODELNAME];	// NB -- this is a WCHAR array for testing the Unicode name function

static wchar_t errmsg[32];						// For the status->string converter
static char cModelName[32];						// buffer for the model name

static string sDevName = "ATN-001";				// device name string
static bool gbWantOneDevice = FALSE;

static int DevNum = 0;				// the device we should work with.
static int DevRange = 1;			// the number of devices we'll send the command to
static int NumDevices = 0;			// used to store the actual number of devices found


// --------------------------- Variables -------------------------------------------------

static int IdleTime = 1;			// default idle time is 1 ms
static int HoldTime = 1;			// default hold time is 1 ms
static int AStart = 0;				// default phase angle start level is 0 degrees.
static int AStop = 63;				// default phase angle stop, for most devices this is 360 degrees so we use that.
static int Dwell = 1000;			// default dwell time is 1 second for first ramp phase
static int Dwell2 = 1000;			// default dwell time is 1 second for second ramp phase (V2 LPS's only)
static int AStep = 1;				// default step size is 1 degree
static int AStep2 = 1;				// default second phase step size for LPS's that support bidirectional ramps

static float WorkingFrequency = 0;	// working frequency
static int PhaseAngle = 0;			// default phase angle is 0 degrees, entered as a floating point value

static int ScaledPhaseAngle = 0;	// temporary storage for scaled phase angle values
static int SerialNumber = 0;		// used to hold the serial number for the get serial number command

static int Sweep_mode = 0;			// used to control the sweep mode
static int GetParam = 0;			// the low byte is the GET command byte

static int ProfileIndex = 0;		// the element in the profile we want to set
static int ProfileLength = 0;		// the length of the profile
static int ProfileValue = 0;		// the profile element's value

static wchar_t ChannelList[256];	// the string listing channel numbers, ranges, or "all"
static int Channel = 1;				// temporary place holder for a single channel
static int NumToRamp = 0;			// number of channels we should ramp (also used as an index -- post increment!)
static int RampMode[64];
static int iRampMode = 0;

bool gbWantSetIdle = FALSE;
bool gbWantSetHold = FALSE;
bool gbWantSetAStart = FALSE;
bool gbWantSetAStop = FALSE;
bool gbWantSetDwell = FALSE;
bool gbWantSetDwell2 = FALSE;
bool gbWantStartSweep = FALSE;
bool gbWantSetAStep = FALSE;
bool gbWantSetAStep2 = FALSE;
bool gbWantSetWorkingFrequency = FALSE;
bool gbWantSetPhaseAngle = FALSE;
bool gbWantSaveSettings = FALSE;
bool gbWantGetParam = FALSE;
bool gbBatchMode = FALSE;
bool gbQuietMode = FALSE;
bool gbWantSetProfileElement = FALSE;
bool gbWantSetProfileLength = FALSE;
bool gbWantChannel = FALSE;
bool gbWantMCRamp = FALSE;
bool gbWantReadSettings = FALSE;





// ------------------------------- Support Routines --------------------------------------

void PrintHelp()
{
	printf("Vaunix Phase Shifter Demonstration\n");
	printf("\n");
	printf("Hit CTRL+C to exit\n");
	printf("\n");

	printf(" --- Overall modes and device selection. Defaults to first device ---\n");
	printf("  -d i n 	Select the devices to work with, i is the device number (1,2,3, etc.)\n");
	printf("     		and n is the number of devices to apply the command to.\n");
	printf("     		-d 1 2 applies the commands to phase shifters 1 and 2.\n");
	printf("     		-d 2 3 applies the commands to phase shifters 2, 3 and 4.\n");
	printf("  -y		Save the current settings in the device for the selected channels.\n");
	printf("\n");
	printf("  -b		Batch mode, exit immediately after sending commands to the Lab Bricks.\n");
	printf("  -q		Quiet mode, skip most outputs.\n");
	printf("  -r		Display information about the device.\n");
	printf("\n");

	printf(" --- Commands to set parameters and start ramp --- \n");
	printf("  -c <list> Set the active channels '1,2,4,5,5-8' or 'all'\n");
	printf("  -f nn     Set working frequency, nn is working frequency in MHz\n");
	printf("  -a nn     Set phase angle, nn is phase angle in 1 degree units\n");
	printf("  -w nn     Set idle time between phase angle ramps, nn is time in ms.\n");
	printf("  -h nn     Set hold time between ramp phases\n");
	printf("  -s nn     Set ramp start value, nn is start value in 1 degree units\n");
	printf("  -e nn     Set ramp end value, nn is end value in 1 degree units\n");
	printf("  -t p nn   Set time to dwell on each phase angle value, nn is time in ms., p is ramp phase 1 or 2\n");

	printf("  -i p nn   Set phase angle ramp increment, nn is the increment\n");
	printf("            in 1 degree units. p is ramp phase 1 or 2\n");

	printf("  -x n rc rc ...	Start multiple ramps, n = number of ramps,  one rc per channel = \n");
	printf("					1 = once upwards, 2 = continuous upwards\n");
	printf("					5 = once down, 6 = continuous down, 17 = bidirectional once,\n");
	printf("					18 = continuous bidirectional ramps, 0 to stop\n");
	printf("					50 = Run profile once, 0 to stop\n");
	printf("\n");


}

// -------------------- - MakeLower------------------------------
wchar_t MakeLowerW(wchar_t &wc)
{
	return wc = towlower(wc);
}

// --------------------------------------------------------------

#define MAX_MSG 32

// -- one way to check for errors --
void CheckAPISet(LVSTATUS status)
{
	if (status & 0x80000000)
	{
		printf("*** Error: LPS API returned status = %x, %s ***\n", status, fnLPS_perror(status));
	}

}


// a helper function that converts a list of channel numbers to a mask in the format used
// by the fnLPS_SetPhaseAngleMC function.
// The channel mask has 1 bits for every channel selected. Its LSB is channel 1, its MSB is channel 64
// The channel list is a string with channel numbers or ranges separated by commas (the function also accepts spaces as delimiters
// but the string from the command line will not have embedded spaces)
// For example, the string 1,3,7 would select channels 1, 3 and 7
// The channel list can also include ranges, the string 1-8 would select channels 1 through 8 inclusive
// If the channel list consists of the word all, then all channels are selected
void ChannelListToMask(unsigned long long * channelmask, wchar_t * channellist)
{
	unsigned long long cmask = 0;
	unsigned long long sel_mask;
	int i;
	int istart, iend;
	wchar_t * p_token;
	wchar_t * p_state;
	wchar_t * p_tmp;

	// change the input string to all lower case
	for (i = 0; i < wcslen(channellist); i++)
	{
		channellist[i] = towlower(channellist[i]);
	}

	// if the user entered "all" select all 64 channels
	// the SetPhaseAngleMC function ignores bits in the mask corresponding to channels it does not have
	if (wcscmp(channellist, L"all") == 0)
	{
		cmask = 0xFFFFFFFFFFFFFFFF;
		*channelmask = cmask;
		return;
	}

	// parse through the string, separating out each entry
	//p_token = wcstok(channellist, L", ", &p_state);
	p_token = wcstok_s(channellist, L", ", &p_state);
	while (p_token != NULL)
	{
		// lets look for a dash to see if we have a range
		p_tmp = wcspbrk(p_token, L"-");
		if (p_tmp != NULL)
		{
			// we have a dash in this token, so it is a range
			*p_tmp = 0;		// by replacing the "-" with a null character we split the token into two strings
			istart = _wtoi(p_token);
			p_tmp++;
			iend = _wtoi(p_tmp);

			// RD -- I don't ignore range inputs where _wtoi returns 0. They are probably typos, but a user
			//		 entering 0 for the first channel accidentally could occur.

			// clip the range start and stop values
			if (istart > 63) istart = 63;
			if (istart < 1) istart = 1;
			if (iend > 64) iend = 64;
			if (iend < 2) iend = 2;

			// add the bits for the selected range to the channel mask
			for (i = istart; i <= iend; i++)
			{
				sel_mask = 1i64 << (i - 1);	// generate the mask bit for this channel number
				cmask |= sel_mask;			// merge it into the channel mask
			}

		}

		// handle a token which is just a channel number
		i = _wtoi(p_token);
		
		if (i != 0)						// ignore any invalid tokens, probably a user typo
		{
			if (i > 64) i = 64;
			if (i < 1) i = 1;			// not really possible due to the filtering of - chars by the range parsing code...
			sel_mask = 1i64 << (i - 1);	// generate the mask bit for this channel number
			cmask |= sel_mask;			// merge it into the channel mask
		}
		p_token = wcstok_s(NULL, L", ", &p_state);
	}

	// return the mask
	*channelmask = cmask;
	return;
}


// ParseCommandLine() will return FALSE to indicate that we received an invalid
// command or should abort for another reason.
bool ParseCommandLine(int argc, _TCHAR *argv[])
{
	int RampPhase;

	enum {
		wantDash, wantDevSubstring, wantIdle, wantAStart, wantAStop, wantDwell, wantAStep,
		wantPhase, wantSweep, wantGetParam, wantDevID, wantDevRange,
		wantDwell2, wantAStep2, wantHold, wantDwellPhase, wantStepPhase, wantWorkingFrequency,
		wantChannel, wantNumToRamp, wantRampCmd
	} state = wantDash;

	for (int i = 1; i < argc; ++i) {
		// Convert each argument to lowercase
		wstring thisParam(argv[i]);
		for_each(thisParam.begin(), thisParam.end(), MakeLowerW);

		if (state == wantDash)
		{
			if ('-' != thisParam[0])
			{
				printf("\n *** Error in command line syntax *** \n");
				PrintHelp();
				return FALSE;
			}
			// remove the dash from the front of the string
			thisParam = wstring(thisParam.begin() + 1, thisParam.end());

			// Identify the command line arguments
			if (L"d" == thisParam) {
				state = wantDevID;
			}
			else if (L"w" == thisParam) {
				gbWantSetIdle = TRUE;
				state = wantIdle;
			}
			else if (L"s" == thisParam) {
				gbWantSetAStart = TRUE;
				state = wantAStart;
			}
			else if (L"e" == thisParam) {
				gbWantSetAStop = TRUE;
				state = wantAStop;
			}
			else if (L"t" == thisParam) {
				state = wantDwellPhase;
			}
			else if (L"i" == thisParam) {
				state = wantStepPhase;
			}
			else if (L"a" == thisParam) {
				gbWantSetPhaseAngle = TRUE;
				state = wantPhase;
			}
			else if (L"g" == thisParam) {
				gbWantStartSweep = TRUE;
				state = wantSweep;
			}
			else if (L"y" == thisParam) {
				gbWantSaveSettings = TRUE;
				state = wantDash;
			}
			else if (L"b" == thisParam) {
				gbBatchMode = TRUE;
				state = wantDash;
			}
			else if (L"q" == thisParam) {
				gbQuietMode = TRUE;
				state = wantDash;
			}
			else if (L"r" == thisParam) {
				gbWantReadSettings = TRUE;
				state = wantDash;
			}
			else if (L"h" == thisParam) {
				gbWantSetHold = TRUE;
				state = wantHold;
			}
			else if (L"f" == thisParam) {
				gbWantSetWorkingFrequency = TRUE;
				state = wantWorkingFrequency;
			}
			else if (L"c" == thisParam) {
				gbWantChannel = TRUE;
				state = wantChannel;
			}
			else if (L"x" == thisParam) {
				gbWantMCRamp = TRUE;
				NumToRamp = 0;
				iRampMode = 0;
				state = wantNumToRamp;
			}

			else {
				// this case is for "-?" and any argument we don't recognize
				PrintHelp();
				return FALSE;	// don't continue
			}
		}

		else {

			// save the whole substring and do conversions for each argument type

			switch (state) {

			case wantDwellPhase:
				RampPhase = _wtoi(thisParam.c_str());
				if (RampPhase == 1) {
					gbWantSetDwell = TRUE;
					state = wantDwell;
				}
				else if (RampPhase == 2) {
					gbWantSetDwell2 = TRUE;
					state = wantDwell2;
				}
				else state = wantDash;		// phase value is wrong, not much we can do about it...
				break;

			case wantStepPhase:
				RampPhase = _wtoi(thisParam.c_str());
				if (RampPhase == 1) {
					gbWantSetAStep = TRUE;
					state = wantAStep;
				}
				else if (RampPhase == 2) {
					gbWantSetAStep2 = TRUE;
					state = wantAStep2;
				}
				else state = wantDash;		// phase value is wrong, not much we can do about it...
				break;

			case wantIdle:
				IdleTime = _wtoi(thisParam.c_str());		// convert to a int
				state = wantDash;
				break;

			case wantHold:
				HoldTime = _wtoi(thisParam.c_str());
				state = wantDash;
				break;

			case wantDevID:
				DevNum = _wtoi(thisParam.c_str());
				state = wantDevRange;
				break;

			case wantChannel:
				ChannelList[255] = 0;	// ensure the ChannelList string is null terminated
				wcsncpy_s(ChannelList, thisParam.c_str(), 255);	// copy limited to buffer size
				state = wantDash;
				break;

			case wantDevRange:
				DevRange = _wtoi(thisParam.c_str());
				state = wantDash;
				break;

			case wantAStart:
				AStart = (int)_wtof(thisParam.c_str());
				state = wantDash;
				break;

			case wantAStop:
				AStop = (int)_wtof(thisParam.c_str());
				state = wantDash;
				break;

			case wantDwell:
				Dwell = _wtoi(thisParam.c_str());
				state = wantDash;
				break;

			case wantDwell2:
				Dwell2 = _wtoi(thisParam.c_str());
				state = wantDash;
				break;

			case wantAStep:
				AStep = (int)_wtof(thisParam.c_str());
				state = wantDash;
				break;

			case wantAStep2:
				AStep2 = (int)_wtof(thisParam.c_str());
				state = wantDash;
				break;

			case wantPhase:
				PhaseAngle = (int)_wtof(thisParam.c_str());	// cast to a float, _wtof actually returns a double
				state = wantDash;
				break;

			case wantWorkingFrequency:
				WorkingFrequency = (float)_wtof(thisParam.c_str());	// cast to a float, _wtof actually returns a double
				state = wantDash;
				break;

			case wantSweep:
				Sweep_mode = _wtoi(thisParam.c_str());
				state = wantDash;
				break;

			case wantNumToRamp:
				NumToRamp = _wtoi(thisParam.c_str());
				if (NumToRamp < 0) NumToRamp = 0;
				if (NumToRamp > 8) NumToRamp = 8;
				state = wantRampCmd;
				break;

			case wantRampCmd:
				RampMode[iRampMode] = _wtoi(thisParam.c_str());
				iRampMode++;
				if (iRampMode < NumToRamp) state = wantRampCmd;
				else state = wantDash;
				break;

			}
		}
	}

	if (state != wantDash) {
		// we are expecting an argument, if we didn't get one then print the help message
		PrintHelp();
		return FALSE;
	}

	// It's OK to continue
	return TRUE;
}
// ------------- Command Line Main ---------------------------------------------------

int _tmain(int argc, _TCHAR* argv[])
{
	int i, j, k;
	int iDev;
	int itemp;
	int itemp2;
	bool bTemp;
	float ftemp;
	int NumChannels;						// temporary storage for the number of channels in a device
	unsigned long long ExChannelMask = 0;	// the mask of selected channels
	unsigned long long TestMask;			// used to test for selected channels in the ExChannelMask
	int ch;

	for (i = 0; i < 64; i++) {
		RampMode[i] = 0;				// initialize the array of ramp mode values for each channel
	}

	if (!ParseCommandLine(argc, argv))
		return 0;

	if (!gbQuietMode) printf("Lab Brick Phase Shifter Multiple Channel Demonstration Program\n");

	//	 -- convert the user's device number to our internal MyDevices array index and check our device range --
	DevNum = DevNum - 1;

	if (DevNum < 0) DevNum = 0;
	if (DevRange < 1) DevRange = 1;
	if (DevRange > MAXDEVICES) DevRange = MAXDEVICES;

	if (DevNum > MAXDEVICES - 1) DevNum = MAXDEVICES - 1;
	if ((DevNum + DevRange) > MAXDEVICES) DevRange = MAXDEVICES - DevNum;

	// at this point our device starting index and number of devices should be reasonable...
	if (!gbQuietMode) printf("Starting device number = %d, using %d device[s]\n", DevNum + 1, DevRange);

	// --- if TestMode = TRUE then the dll will fake the hardware ---
	fnLPS_SetTestMode(FALSE);


	// --- Use the tracing control function to turn on debug messages
	fnLPS_SetTraceLevel(0, 0, false);
	// fnLPS_SetTraceLevel(3, 3, false);

	i = fnLPS_GetNumDevices();

	if (i == 0) {
		printf("No device found\n");
	}

	if (i == 1) {
		if (!gbQuietMode) printf("Found %d Device\n", i);

	}
	else {
		if (!gbQuietMode) printf("Found %d Devices\n", i);
	}

	// -- warn the user if he or she expects more devices than we have --
	if (DevRange > i) {
		printf(" Warning - not enough phase shifters are connected\n");
	}

	NumDevices = fnLPS_GetDevInfo(MyDevices);

	if (!gbQuietMode) printf("Got Device Info for %d Device[s]\n", NumDevices);


	if (NumDevices > 0)	// do we have a device?
	{
		for (j = 0; j < NumDevices; j++) {
			printf("\n---------- Device %d ------------------------------\n", j + 1);

			// --- print out the first device's name ---
			if (!gbQuietMode) {
				itemp = fnLPS_GetModelNameA(MyDevices[j], MyDeviceNameA);
				printf("Device %d is an %s \n", MyDevices[j], MyDeviceNameA);
			}

			// --- print out the device's serial number ---
			if (!gbQuietMode) {
				itemp = fnLPS_GetSerialNumber(MyDevices[j]);
				if (itemp >= 0)
					printf("Device %d has serial number %d \n", MyDevices[j], itemp);
			}


			// --- We need to init the device (open it) before we can do anything else ---
			itemp = fnLPS_InitDevice(MyDevices[j]);

			if (itemp) {
				printf("InitDevice returned error code %x\n", itemp);
			}

			fnLPS_SetTraceLevel(0, 0, false);

			// --- Lets see if we got the device's parameters ---
			if (!gbQuietMode && gbWantReadSettings) {

				// Display the number of channels for this device
				NumChannels = fnLPS_GetNumChannels(MyDevices[j]);
				if (NumChannels >= 0) {
					if (NumChannels == 1)
					{
						printf("Single Channel Device\n");
					}
					else printf("Device has %d Channels\n", NumChannels);
				}
				else
					CheckAPISet(NumChannels);

				// show device wide parameters
				printf("Phase Angle Range:\n");
				itemp = fnLPS_GetMinPhaseShift(MyDevices[j]);
				if (itemp >= 0)
					printf("Minimum Phase Shift = %d degrees\n", itemp);
				else
					CheckAPISet(itemp);

				itemp = fnLPS_GetMaxPhaseShift(MyDevices[j]);
				if (itemp >= 0)
					printf("Maximum Phase Shift = %d degrees\n", itemp);
				else
					CheckAPISet(itemp);

				if (fnLPS_GetFeatures(MyDevices[j]) & HAS_HIRES)
				{
					printf("Working Frequency Range:\n");
					itemp = fnLPS_GetMinWorkingFrequency(MyDevices[j]);
					ftemp = ((float)itemp) / 10;		// frequency is in 100KHz units
					if (itemp >= 0)
						printf("Minimum Frequency = %.2f Mhz\n", ftemp);
					else
						CheckAPISet(itemp);

					itemp = fnLPS_GetMaxWorkingFrequency(MyDevices[j]);
					ftemp = ((float)itemp) / 10;		// frequency is in 100KHz units
					if (itemp >= 0)
						printf("Maximum Frequency = %.2f Mhz\n", ftemp);
					else
						CheckAPISet(itemp);
				}

				// if we have more than one channel, show the parameters for each channel
				for (i = 0; i < NumChannels; i++)
				{
					if (NumChannels > 1)
					{
						printf("---------- Parameters for Channel %d/%d ------\n", i + 1, NumChannels);
						fnLPS_SetChannel(MyDevices[j], i + 1);	// the channel argument runs from 1 to N channels
						CheckAPISet(itemp);
					}

					// Display the frequency related parameters
					itemp = fnLPS_GetWorkingFrequency(MyDevices[j]);
					ftemp = ((float)itemp) / 10;	// working frequency is in 100KHz units
					if (itemp >= 0)
						printf("Working Frequency = %.2f Mhz\n", ftemp);
					else
						CheckAPISet(itemp);
					
					itemp = fnLPS_GetPhaseAngle(MyDevices[j]);
					if (itemp >= 0)
						printf("Phase Angle = %d degrees\n", itemp);
					else
						CheckAPISet(itemp);

					itemp = fnLPS_GetRampStart(MyDevices[j]);
					if (itemp >= 0)
						printf("Ramp Start Level = %d degrees\n", itemp);
					else
						CheckAPISet(itemp);

					itemp = fnLPS_GetRampEnd(MyDevices[j]);
					if (itemp >= 0)
						printf("Ramp End Level = %d degrees\n", itemp);
					else
						CheckAPISet(itemp);

					itemp = fnLPS_GetPhaseAngleStep(MyDevices[j]);
					if (itemp >= 0)
						printf("First Phase Ramp Phase Angle Step Size = %d degrees\n", itemp);
					else
						CheckAPISet(itemp);

					if (fnLPS_GetFeatures(MyDevices[j]) > 0)
					{
						itemp = fnLPS_GetPhaseAngleStepTwo(MyDevices[j]);
						if (itemp >= 0)
							printf("Second Phase Ramp Phase Angle Step Size = %d degrees\n", itemp);
						else
							CheckAPISet(itemp);
					}

					itemp = fnLPS_GetDwellTime(MyDevices[j]);
					if (itemp >= 0)
						printf("First Phase Ramp Dwell Time = %d ms\n", itemp);
					else
						CheckAPISet(itemp);

					if (fnLPS_GetFeatures(MyDevices[j]) > 0)
					{
						itemp = fnLPS_GetDwellTimeTwo(MyDevices[j]);
						if (itemp >= 0)
							printf("Second Phase Ramp Dwell Time = %d ms\n", itemp);
						else
							CheckAPISet(itemp);
					}

					itemp = fnLPS_GetIdleTime(MyDevices[j]);
					if (itemp >= 0)
						printf("Ramp Idle Time = %d ms\n", itemp);
					else
						CheckAPISet(itemp);

					if (fnLPS_GetFeatures(MyDevices[j]) > 0)
					{
						itemp = fnLPS_GetHoldTime(MyDevices[j]);
						if (itemp >= 0)
							printf("Ramp Hold Time = %d ms\n", itemp);
						else
							CheckAPISet(itemp);
					}

				} // end of our loop over channels
				printf("\n");

			} // end of our quiet mode read the settings case

		} // end of the for loop over the devices

		// if the user is trying to control a device we don't have, then quit now
		if (DevNum > NumDevices - 1) {
			for (j = 0; j < NumDevices; j++)
			{
				itemp = fnLPS_CloseDevice(MyDevices[j]);
			}
			printf("First selected device is not attached, exiting.\n");
			return 0;			// quit - nothing else to do
		}

		// if the user is trying to control more devices than we have, reduce the number of devices in the group
		if ((DevNum + DevRange) > NumDevices) {
			DevRange = NumDevices - DevNum;
			printf("Not enough phase shifters connected, using %d devices.\n", DevRange);
		}

		// ------------- Now we'll set the requested device or devices with new parameters -------------
		if (!gbQuietMode)printf("Setting the phase shifter parameters..\n");

		for (iDev = DevNum; iDev < DevNum + DevRange; iDev++) {

			// For this example we generate a 64 bit wide mask with a 1 bit for each channel
			// that the set phase angle command should change

			if (gbWantChannel)
			{
				ChannelListToMask(&ExChannelMask, ChannelList);

				if (!gbQuietMode) printf("ExtendedChannelMask is %llx \n", ExChannelMask);
			}

			// --- first we'll get the number of channels for this device
			NumChannels = fnLPS_GetNumChannels(MyDevices[iDev]);
			if (NumChannels == 1)
			{
				ExChannelMask = 0x1;
			}
			else
			{
				if (ExChannelMask == 0)
				{
					if (!gbQuietMode) printf("No Channels Selected\n");
				}
			}

			// --- Lets set the phase angle first using the multiple channel API function ---
			if (gbWantSetPhaseAngle) {

				ScaledPhaseAngle = PhaseAngle;

				// Set the selected channels with the phase angle
				if (!gbQuietMode) printf("Setting the phase angle for channel in ExtendedChannelMask %llx to %d degres\n", ExChannelMask, ScaledPhaseAngle);
				itemp = fnLPS_SetPhaseAngleMC(MyDevices[iDev], ScaledPhaseAngle, ExChannelMask);

			}

			// --- and then do whatever else the user requested, looping over the selected channels
			TestMask = 1;

			if ((NumChannels <= 0) || (NumChannels > 64)) NumChannels = 1;	// defend against bad values

			for (ch = 0; ch < NumChannels; ch++)
			{
				if (TestMask & ExChannelMask)
				{
					// set the channel
					fnLPS_SetChannel(MyDevices[iDev], ch + 1);	// the channel argument runs from 1 to N channels

					if (gbWantSetDwell) {

						if (!gbQuietMode) printf("Setting the first phase dwell time for device %d channel %d to %d ms\n", iDev + 1, ch + 1, Dwell);
						itemp = fnLPS_SetDwellTime(MyDevices[iDev], Dwell);
						CheckAPISet(itemp);
					}

					if (gbWantSetAStart) {

						if (!gbQuietMode) printf("Setting the ramp start for device %d channel %d to %d degrees\n", iDev + 1, ch + 1, AStart);
						itemp = fnLPS_SetRampStart(MyDevices[iDev], AStart);
						CheckAPISet(itemp);
					}

					if (gbWantSetAStop) {

						if (!gbQuietMode) printf("Setting ramp end for device %d channel %d to %d degrees\n", iDev + 1, ch + 1, AStop);
						itemp = fnLPS_SetRampEnd(MyDevices[iDev], AStop);
						CheckAPISet(itemp);
					}

					if (gbWantSetAStep) {

						if (!gbQuietMode) printf("Setting the first phase phase angle step for device %d channel %d to %d degrees\n", iDev + 1, ch + 1, AStep);
						itemp = fnLPS_SetPhaseAngleStep(MyDevices[iDev], AStep);
						CheckAPISet(itemp);
					}

					if (gbWantSetIdle) {

						if (!gbQuietMode) printf("Setting the idle time between ramps for device %d channel %d to %d ms\n", iDev + 1, ch + 1, IdleTime);
						itemp = fnLPS_SetIdleTime(MyDevices[iDev], IdleTime);
						CheckAPISet(itemp);
					}


					// if we have a V2 Lab Brick, send it the additional commands
					if (fnLPS_GetFeatures(MyDevices[iDev]) > 0)
					{
						if (gbWantSetAStep2)
						{
							if (!gbQuietMode) printf("Setting the second phase phase angle step for device %d channel %d to %d degrees\n", iDev + 1, ch + 1, AStep2);
							itemp = fnLPS_SetPhaseAngleStepTwo(MyDevices[iDev], AStep2);
							CheckAPISet(itemp);
						}
						if (gbWantSetDwell2)
						{
							if (!gbQuietMode) printf("Setting the second phase dwell time for device %d channel %d to %d ms\n", iDev + 1, ch + 1, Dwell2);
							itemp = fnLPS_SetDwellTimeTwo(MyDevices[iDev], Dwell2);
							CheckAPISet(itemp);
						}
						if (gbWantSetHold)
						{
							if (!gbQuietMode) printf("Setting the hold time between ramp phases for device %d channel %d to %d ms\n", iDev + 1, ch + 1, HoldTime);
							itemp = fnLPS_SetHoldTime(MyDevices[iDev], HoldTime);
							CheckAPISet(itemp);
						}

						// RD 1-14-25 cleaning up a needless message..
						if (gbWantSetWorkingFrequency)
						{
							// make sure the desired working frequency is in range for this device
							itemp = fnLPS_GetMinWorkingFrequency(MyDevices[iDev]);
							CheckAPISet(itemp);
							itemp2 = (int)(WorkingFrequency * 10);					// converting from Mhz to 100KHz integer units

							// clip the min value for this device and let the user know
							if ((itemp >= 0) && (itemp2 < itemp))
							{
								if (!gbQuietMode) printf("Working frequency for device %d of %.2f Mhz too low\n", iDev + 1, WorkingFrequency);
								itemp2 = itemp;
							}

							itemp = fnLPS_GetMaxWorkingFrequency(MyDevices[iDev]);
							CheckAPISet(itemp);

							// clip the max value for this device and let the user know
							if ((itemp >= 0) && (itemp2 > itemp))
							{
								if (!gbQuietMode) printf("Working frequency for device %d of %.2f Mhz too high\n", iDev + 1, WorkingFrequency);
								itemp2 = itemp;
							}

							ftemp = (float)(itemp2) / 10;

							if (!gbQuietMode) printf("Setting the working frequency for device %d channel %d to %.2f Mhz\n", iDev + 1, Channel, ftemp);

							itemp = fnLPS_SetWorkingFrequency(MyDevices[iDev], itemp2);
							CheckAPISet(itemp);

						}
					}

					if (gbWantSaveSettings)
					{
						if (!gbQuietMode) printf("Saving the settings for device %d channel %d\n", iDev + 1, ch + 1);
						itemp = fnLPS_SaveSettings(MyDevices[iDev]);
						CheckAPISet(itemp);
						Sleep(100);
					}

				}	// end of if active channel bit test

				TestMask <<= 1;		// move our test bit over to test the next channel's selection bit

			}	// end of loop over NumChannels for this device
		}	// this is the end of our for loop over devices for the general commands

		
		// --- Now we will handle the case of multi-channel ramps, controlled by the channel list ---
		if (!gbQuietMode) printf("\n");
		// -- For multi channel ramps we first set the parameters, then send the actual commands to start the ramps
		//	  grouping these commands reduces the latency between the ramps on each phase shifter
		//	  Note that this command uses the same per channel settings for each device

		for (iDev = DevNum; iDev < DevNum + DevRange; iDev++)
		{
			if (gbWantMCRamp)
			{
				// --- first we'll get the number of channels for this device
				NumChannels = fnLPS_GetNumChannels(MyDevices[iDev]);
				if ((NumChannels <= 0) || (NumChannels > 64)) NumChannels = 1;	// defend against bad values

				// --- then we loop over the channels, setting the ramp parameters from our array
				//     for each channel in our channel selection mask (and therefore in the channel selection list)
				TestMask = 1;
				for (ch = 0; ch < NumChannels; ch++)
				{
					if (TestMask & ExChannelMask)
					{
						fnLPS_SetChannel(MyDevices[iDev], ch + 1);	// the channel argument runs from 1 to N channels

						if (RampMode[ch] != 0)
						{
							// --- The user wants to start some kind of an phase angle ramp ---
							if (RampMode[ch] & CL_SWP_DIRECTION)
							{
								bTemp = FALSE;
							}
							else
							{
								bTemp = TRUE;
							}	// NB -- the flag is TRUE for "up" in the Set...Direction call.
								// but the old test program uses a 0 bit for up, and a 1 bit for down...

							itemp = fnLPS_SetRampDirection(MyDevices[iDev], bTemp);
							CheckAPISet(itemp);

							// --- and now we'll do the mode - one time or repeated ---
							if (RampMode[ch] & CL_SWP_ONCE)
							{
								bTemp = FALSE;
							}
							else
							{
								bTemp = TRUE;
							}	// NB -- the flag is TRUE for "repeated" in the SetSweepMode call.
							// but the old test program encodes the modes differently

							itemp = fnLPS_SetRampMode(MyDevices[iDev], bTemp);
							CheckAPISet(itemp);

							// --- and then the bidirectional ramp control if the device is a V2 device
							if (fnLPS_GetFeatures(MyDevices[iDev]) > 0)
							{
								if (RampMode[ch] & CL_SWP_BIDIRECTIONALLY)
								{
									bTemp = TRUE;
								}							// the command line has true for bidirectional 
								else						// as does the actual HW command...
								{
									bTemp = FALSE;
								}

								printf("Bidirection mode set to %x \n", bTemp);
								itemp = fnLPS_SetRampBidirectional(MyDevices[iDev], bTemp);
								CheckAPISet(itemp);
							}

						}
					}

					TestMask <<= 1;		// move to the next channel
				}



				// --- then we loop over the channels and launch the ramps ---
				TestMask = 1;

				for (ch = 0; ch < NumChannels; ch++)
				{
					// start or stop the ramps for the selected channels 
					if (TestMask & ExChannelMask)
					{
						fnLPS_SetChannel(MyDevices[iDev], ch + 1);	// the channel argument runs from 1 to N channels

						if (RampMode[ch] == 0)
						{
							if (!gbQuietMode) printf("Stopping the Phase Angle Ramp for device %d channel %d\n", iDev + 1, ch + 1);
							itemp = fnLPS_StartRamp(MyDevices[iDev], FALSE);
							CheckAPISet(itemp);
						}
						else
						{
							if (!gbQuietMode) printf("Starting a phase angle ramp for device %d, channel %d\n", iDev + 1, ch + 1);
							itemp = fnLPS_StartRamp(MyDevices[iDev], TRUE);
							CheckAPISet(itemp);
						}
					}

					TestMask <<= 1;		// move to the next channel
				}
			}
		} // this is the end of our for loop over selected devices for the multi channel ramp command


		// -- Lets report on the device's operation for a little while, unless we are in batch mode
		if (!gbBatchMode)
		{

			j = 0;
			while (j < 30)
			{
				for (iDev = DevNum; iDev < DevNum + DevRange; iDev++)
				{
					// use the HiRes function and show all the channels for the device
					NumChannels = fnLPS_GetNumChannels(MyDevices[iDev]);
					if (NumChannels <= 0) NumChannels = 1;	// protect against an error return
					for (k = 1; k < NumChannels + 1; k++)
					{
						if (NumChannels > 1) {
							fnLPS_SetChannel(MyDevices[iDev], k);
						}

						// show the per channel status for ramps and profiles
						itemp = fnLPS_GetDeviceStatus(MyDevices[iDev]);
						//printf("GetDeviceStatus returned: %x ", itemp);

						if (itemp & SWP_ACTIVE)
						{
							printf("Ramp in progress on channel %d\n", k);
						}
						if (itemp & PROFILE_ACTIVE)
						{
							printf("Profile in progress on channel %d\n", k);
						}

						itemp = fnLPS_GetPhaseAngle(MyDevices[iDev]);
						printf("Phase Angle = %d degrees for device %d, channel %d\n", itemp, iDev + 1, k);

					}
				}
				printf("\n");
				Sleep(100);		// wait for 0.1  second
				j++;
			}

		} // end of if not batch mode

		// -- we've done whatever the user wanted, time to close the devices
		printf("Closing devices...\n");
		for (j = 0; j < NumDevices; j++)
		{
			itemp = fnLPS_CloseDevice(MyDevices[j]);
		}
		printf("Done.\n");

	} // end of if ( i > 0 ) -- "we have a device"

	return 0;
}

// ===================== end of main ======================================
